<?php
/**
 * @package TSF_Extension_Manager\Extension\Monitor\Monitor\Data
 */

namespace TSF_Extension_Manager\Extension\Monitor;

\defined( 'TSF_EXTENSION_MANAGER_PRESENT' ) or die;

if ( \tsfem()->_blocked_extension_file( $_instance, $bits[1] ) ) return;

/**
 * Monitor extension for The SEO Framework
 * Copyright (C) 2016 - 2024 Sybre Waaijer, CyberWire B.V. (https://cyberwire.nl/)
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Class TSF_Extension_Manager\Extension\Monitor\Data
 *
 * Holds extension data methods.
 *
 * @since 1.0.0
 * @access private
 * @errorval 101xxxx
 * @uses TSF_Extension_Manager\Traits
 */
class Data {
	use \TSF_Extension_Manager\Construct_Core_Once_Interface;

	/**
	 * Constructor. Verifies integrity.
	 *
	 * @since 1.0.0
	 */
	private function construct() {
		// Verify integrity.
		$that = __NAMESPACE__ . ( \is_admin() ? '\\Admin' : '\\Front' );
		$this instanceof $that or \wp_die( -1 );
	}

	/**
	 * Ticks data caches to see if there's an update required.
	 * Loaded early to set notices in time.
	 *
	 * @since 1.1.0
	 */
	protected function prepare_data() {
		if ( ! $this->get_option( 'monitor_installing', false ) ) {
			$this->get_data( 'issues', [] );
		}
	}

	/**
	 * Returns Monitor Data.
	 *
	 * @since 1.0.0
	 *
	 * @param string $type    The monitor data type. Accepts 'issue'.
	 * @param mixed  $default The fallback data to return if no data is found.
	 * @return array|mixed The found data.
	 */
	protected function get_data( $type, $default = null ) {

		/**
		 * Return null if this is the first run; to eliminate duplicated calls
		 * to the API server. Which would otherwise return "not found" data anyway.
		 */
		if ( $this->get_option( 'monitor_installing', false ) ) {
			$this->set_installing_site( false );
			return null;
		}

		/**
		 * @see trait TSF_Extension_Manager\Extension_Options
		 */
		return $this->get_option( $type, [] )
			?: $this->get_remote_data( $type )
			?: $default;
	}

	/**
	 * Returns Monitor Data fetched externally from the API server.
	 * If no locally stored data is found, new data gets fetched.
	 *
	 * @since 1.0.0
	 *
	 * @param string $type The monitor data type. Accepts 'issue'.
	 * @return array|boolean The found data. False on failure.
	 */
	protected function get_remote_data( $type = '' ) {

		if ( ! $type )
			return false;

		$this->is_remote_data_expired() and $this->api_get_remote_data();

		/**
		 * Option cache should be updated.
		 *
		 * @see trait TSF_Extension_Manager\Extension_Options
		 */
		return $this->get_option( $type, [] );
	}

	/**
	 * Deletes data fetched remotely.
	 *
	 * @since 1.0.0
	 */
	protected function delete_data() {
		$this->delete_option( 'issues' );
	}
}
