<?php
/**
 * @package TSF_Extension_Manager\Extension\Transport\Importers
 */

namespace TSF_Extension_Manager\Extension\Transport\Importers\TermMeta;

\defined( 'TSFEM_E_TRANSPORT_VERSION' ) or die;

/**
 * Transport extension for The SEO Framework
 * copyright (C) 2022 - 2024 Sybre Waaijer, CyberWire B.V. (https://cyberwire.nl/)
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Importer for Yoast SEO.
 *
 * @since 1.0.0
 * @access private
 *
 * Inherits abstract setup_vars.
 */
final class WordPress_SEO extends Base {

	/**
	 * Sets up variables.
	 *
	 * @since 1.0.0
	 * @abstract
	 */
	protected function setup_vars() {
		global $wpdb;

		// phpcs:disable, WordPress.Arrays.MultipleStatementAlignment -- deeply nested is still simple here.

		// Construct and fetch classname.
		$transformer_class = \get_class(
			\TSF_Extension_Manager\Extension\Transport\Transformers\WordPress_SEO::get_instance()
		);

		/**
		 * [ $from_table, $from_index ]
		 * [ $to_table, $to_index ]
		 * $transformer
		 * $sanitizer
		 * $transmuter
		 * $cb_after_loop
		 */
		$this->conversion_sets = [
			[
				null,
				[ $wpdb->termmeta, \THE_SEO_FRAMEWORK_TERM_OPTIONS ],
				null,
				null,
				[
					'name'    => 'Yoast SEO Term Meta',
					'from' => [
						[ $this, '_get_wpseo_transport_term_ids' ],
						[ $this, '_get_wpseo_term_transport_value' ],
					],
					'to'      => [
						null,
						[ $this, '_wpseo_term_meta_transmuter' ],
					],
					'to_data' => [
						'transmuters'  => [
							'wpseo_title'                 => 'doctitle',
							'wpseo_desc'                  => 'description',
							'wpseo_opengraph-title'       => 'og_title',
							'wpseo_opengraph-description' => 'og_description',
							'wpseo_opengraph-image'       => 'social_image_url',
							'wpseo_opengraph-image-id'    => 'social_image_id',
							'wpseo_twitter-title'         => 'tw_title',
							'wpseo_twitter-description'   => 'tw_description',
							'wpseo_canonical'             => 'canonical',
							'wpseo_noindex'               => 'noindex',
						],
						'transformers' => [
							'wpseo_title'                 => [ $transformer_class, '_title_syntax' ],
							'wpseo_desc'                  => [ $transformer_class, '_description_syntax' ],
							'wpseo_opengraph-title'       => [ $transformer_class, '_title_syntax' ],
							'wpseo_opengraph-description' => [ $transformer_class, '_description_syntax' ],
							'wpseo_twitter-title'         => [ $transformer_class, '_title_syntax' ],
							'wpseo_twitter-description'   => [ $transformer_class, '_description_syntax' ],
							'wpseo_noindex'               => [ $transformer_class, '_robots_text_to_qubit' ], // also sanitizes
						],
						'sanitizers' => [
							'wpseo_title'                 => 'TSF_Extension_Manager\Transition\sanitize_metadata_content',
							'wpseo_desc'                  => 'TSF_Extension_Manager\Transition\sanitize_metadata_content',
							'wpseo_opengraph-title'       => 'TSF_Extension_Manager\Transition\sanitize_metadata_content',
							'wpseo_opengraph-description' => 'TSF_Extension_Manager\Transition\sanitize_metadata_content',
							'wpseo_opengraph-image'       => 'sanitize_url',
							'wpseo_opengraph-image-id'    => 'absint',
							'wpseo_twitter-title'         => 'TSF_Extension_Manager\Transition\sanitize_metadata_content',
							'wpseo_twitter-description'   => 'TSF_Extension_Manager\Transition\sanitize_metadata_content',
							'wpseo_canonical'             => 'sanitize_url',
						],
					],
				],
				[ $this, '_term_meta_option_cleanup' ],
			],
		];
		// phpcs:enable, WordPress.Arrays.MultipleStatementAlignment
	}

	/**
	 * Returns data from wpseo_taxonomy_meta.
	 *
	 * @since 1.0.0
	 *
	 * @return array Yoast SEO data: {
	 *     string $taxonomy => array $terms {
	 *        int $term_id => array $meta {
	 *           ?string $meta_index => ?mixed $meta_value
	 *        }
	 *     }
	 * }
	 */
	private function get_wpseo_taxonomy_meta() {
		static $data;
		return $data ??= \get_option( 'wpseo_taxonomy_meta' ) ?: [];
	}

	/**
	 * Obtains ids from Yoast SEO's taxonomy metadata.
	 *
	 * @since 1.0.0
	 *
	 * @param array $data Any useful data pertaining to the current transmutation type.
	 * @return array|null Array if existing values are present, null otherwise.
	 */
	protected function _get_wpseo_transport_term_ids( $data ) {

		$ids = [];

		foreach ( $this->get_wpseo_taxonomy_meta() as $data )
			$ids = array_merge( $ids, array_keys( $data ) );

		return $ids;
	}

	/**
	 * Returns existing advanced robots values.
	 *
	 * @since 1.0.0
	 *
	 * @param array  $data    Any useful data pertaining to the current transmutation type.
	 * @param array  $actions The actions for and after transmuation, passed by reference.
	 * @param array  $results The results before and after transmuation, passed by reference.
	 * @param ?array $cleanup The extraneous database indexes to clean up, passed by reference.
	 * @throws \Exception On database error when \WP_DEBUG is enabled.
	 * @return array|null Array if existing values are present, null otherwise.
	 */
	protected function _get_wpseo_term_transport_value( $data, &$actions, &$results, &$cleanup ) {

		// No need to access the index a dozen times, store pointer in var.
		$item_id = &$data['item_id'];

		// Walk until index is found. We are unaware of taxonomies during transportation.
		foreach ( $this->get_wpseo_taxonomy_meta() as $taxonomy => $meta ) {
			if ( \array_key_exists( $item_id, $meta ) ) {
				$transport_value = $meta[ $item_id ];
				break;
			}
		}

		return $transport_value ?? null;
	}

	/**
	 * Transmutes Yoast SEO Meta to TSF's serialized metadata.
	 *
	 * @since 1.0.0
	 * @generator
	 *
	 * @param array  $data    Any useful data pertaining to the current transmutation type.
	 * @param ?array $actions The actions for and after transmuation, passed by reference.
	 * @param ?array $results The results before and after transmutation, passed by reference.
	 * @throws \Exception On database error when \WP_DEBUG is enabled.
	 */
	protected function _wpseo_term_meta_transmuter( $data, &$actions, &$results ) {

		[ $from_table, $from_index ] = $data['from'];
		[ $to_table, $to_index ]     = $data['to'];

		$set_value = [];

		// Nothing to do here, TSF already has value set. Skip to next item.
		if ( ! $actions['transport'] ) goto useless;

		foreach ( $data['to_data']['transmuters'] as $from => $to ) {
			$_set_value = $data['set_value'][ $from ] ?? null;

			// We assume here that all Rank Math data without value is useless.
			// This might prove an issue later, where 0 carries significance.
			// Though, no developer in their right mind would store 0 or empty string... Oh wait, we're dealing with Yoast.
			if ( \in_array( $_set_value, $this->useless_data, true ) ) continue;

			$_transformed = 0;

			if ( isset( $data['to_data']['transformers'][ $from ] ) ) {
				$_pre_transform_value = $_set_value;

				$_set_value = \call_user_func_array(
					$data['to_data']['transformers'][ $from ],
					[
						$_set_value,
						$data['item_id'],
						$this->type,
						[ $from_table, $from_index ],
						[ $to_table, $to_index ],
					]
				);

				// We actually only read this as boolean. Still, might be fun later.
				$_transformed = (int) ( $_pre_transform_value !== $_set_value );
			}

			if ( isset( $data['to_data']['sanitizers'][ $from ] ) ) {
				$_pre_sanitize_value   = $_set_value;
				$_set_value            = \call_user_func( $data['to_data']['sanitizers'][ $from ], $_set_value );
				$results['sanitized'] += (int) ( $_pre_sanitize_value !== $set_value );
			}

			if ( ! \in_array( $_set_value, $this->useless_data, true ) ) {
				$set_value[ $to ]        = $_set_value;
				$results['transformed'] += $_transformed;

				// If the title is not useless, assume it must remain how the user set it.
				if ( 'doctitle' === $to )
					$set_value['title_no_blog_name'] = 1;
			}
		}

		if ( \in_array( $set_value, $this->useless_data, true ) ) {
			useless:;
			$set_value              = null;
			$actions['transport']   = false;
			$results['transformed'] = 0;
		}

		$this->transmute(
			$set_value,
			$data['item_id'],
			[ $from_table, $from_index ], // Should be [ null, null ]
			[ $to_table, $to_index ],
			$actions,
			$results
		);

		yield 'transmutedResults' => [ $results, $actions ];
	}

	/**
	 * Cleans Yoast SEO Meta from the database.
	 *
	 * @since 1.0.0
	 * @generator
	 *
	 * @param array $item_ids The term IDs looped over.
	 * @return ?void Early when option is not registered.
	 */
	protected function _term_meta_option_cleanup( $item_ids ) {

		// If no items are looped over, test if the option even exists before deleting.
		// If option is false, $item_ids is always empty, but not vice versa. Saved db call.
		if ( ! $item_ids ) {
			global $wpdb;

			if ( null === $wpdb->get_var(
				$wpdb->prepare(
					"SELECT option_value FROM `$wpdb->options` WHERE option_name = %s",
					'wpseo_taxonomy_meta'
				)
			) ) return;
		}

		yield 'afterResults' => [
			// This also invokes all necessary cache clearning. This function runs only once.
			(bool) \delete_option( 'wpseo_taxonomy_meta' ), // assert success
			[ // onsuccess
				'message' => \__( 'Cleanup: Deleted old term meta successfully.', 'the-seo-framework-extension-manager' ),
				'addTo'   => 'deleted', // writes variable, must never be untrusted
				'count'   => \count( $item_ids ), // Success "count."
			],
			[ // onfailure
				'message' => \__( 'Cleanup: Failed to delete old term meta.', 'the-seo-framework-extension-manager' ),
				'addTo'   => 'failed', // writes variable, must never be untrusted
				'count'   => 1, // "failure" count
			],
		];
	}
}
